import { type NextRequest, NextResponse } from "next/server"
import { query } from "@/lib/db"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const tenantId = searchParams.get("tenant_id")

    if (!tenantId) {
      return NextResponse.json({ error: "Tenant ID is required" }, { status: 400 })
    }

    const result = await query(
      `
      SELECT 
        sl.stock_id,
        sl.quantity_on_hand,
        sl.quantity_reserved,
        sl.quantity_available,
        sl.reorder_level,
        p.product_code,
        p.product_name,
        pv.variant_code,
        pv.variant_name,
        pv.cost_price,
        pv.selling_price,
        w.warehouse_name,
        (sl.quantity_on_hand * pv.cost_price) as stock_value,
        CASE 
          WHEN sl.quantity_available <= sl.reorder_level THEN true 
          ELSE false 
        END as is_low_stock
      FROM stock_level sl
      JOIN product_variant pv ON sl.variant_id = pv.variant_id
      JOIN product p ON pv.product_id = p.product_id
      JOIN warehouse w ON sl.warehouse_id = w.warehouse_id
      WHERE sl.sys_tenant_id = $1
      ORDER BY p.product_name, pv.variant_name, w.warehouse_name
    `,
      [tenantId],
    )

    return NextResponse.json({
      inventory: result.rows,
    })
  } catch (error) {
    console.error("Inventory API error:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}
