import { type NextRequest, NextResponse } from "next/server"
import { query } from "@/lib/db"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const tenantId = searchParams.get("tenant_id")

    if (!tenantId) {
      return NextResponse.json({ error: "Tenant ID is required" }, { status: 400 })
    }

    // Get total products
    const productsResult = await query(
      `SELECT COUNT(DISTINCT pv.variant_id) as total_products
       FROM product_variant pv 
       WHERE pv.sys_tenant_id = $1 AND pv.is_active = true`,
      [tenantId],
    )

    // Get low stock items
    const lowStockResult = await query(
      `SELECT COUNT(*) as low_stock_items
       FROM stock_level sl
       WHERE sl.sys_tenant_id = $1 AND sl.quantity_available <= sl.reorder_level`,
      [tenantId],
    )

    // Get total inventory value
    const valueResult = await query(
      `SELECT COALESCE(SUM(sl.quantity_on_hand * pv.cost_price), 0) as total_value
       FROM stock_level sl
       JOIN product_variant pv ON sl.variant_id = pv.variant_id
       WHERE sl.sys_tenant_id = $1`,
      [tenantId],
    )

    // Get total warehouses
    const warehousesResult = await query(
      `SELECT COUNT(*) as total_warehouses
       FROM warehouse w
       WHERE w.sys_tenant_id = $1 AND w.is_active = true`,
      [tenantId],
    )

    return NextResponse.json({
      stats: {
        totalProducts: Number.parseInt(productsResult.rows[0].total_products) || 0,
        lowStockItems: Number.parseInt(lowStockResult.rows[0].low_stock_items) || 0,
        totalValue: Number.parseFloat(valueResult.rows[0].total_value) || 0,
        totalWarehouses: Number.parseInt(warehousesResult.rows[0].total_warehouses) || 0,
      },
    })
  } catch (error) {
    console.error("Inventory Stats API error:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}
