import { type NextRequest, NextResponse } from "next/server"
import { query } from "@/lib/db"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const tenantId = searchParams.get("tenant_id")

    if (!tenantId) {
      return NextResponse.json({ error: "Tenant ID is required" }, { status: 400 })
    }

    const result = await query(
      `
      SELECT 
        po.po_id,
        po.po_number,
        po.po_date,
        po.expected_delivery_date,
        po.status,
        po.total_amount,
        po.created_at,
        s.supplier_name,
        u.first_name || ' ' || u.last_name as created_by_name
      FROM purchase_order po
      JOIN supplier s ON po.supplier_id = s.supplier_id
      JOIN sys_user u ON po.created_by = u.user_id
      WHERE po.sys_tenant_id = $1
      ORDER BY po.created_at DESC
    `,
      [tenantId],
    )

    return NextResponse.json({
      purchaseOrders: result.rows,
    })
  } catch (error) {
    console.error("Purchase Orders API error:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { supplier_id, warehouse_id, po_date, expected_delivery_date, line_items, notes, created_by, sys_tenant_id } =
      body

    // Generate PO number
    const poNumberResult = await query(
      `
      SELECT 'PO-' || TO_CHAR(CURRENT_DATE, 'YYYY') || '-' || 
             LPAD((COALESCE(MAX(CAST(SUBSTRING(po_number FROM 9) AS INTEGER)), 0) + 1)::TEXT, 3, '0') as po_number
      FROM purchase_order 
      WHERE sys_tenant_id = $1 AND po_number LIKE 'PO-' || TO_CHAR(CURRENT_DATE, 'YYYY') || '-%'
    `,
      [sys_tenant_id],
    )

    const poNumber = poNumberResult.rows[0].po_number

    // Calculate totals
    const subtotal = line_items.reduce((sum: number, item: any) => sum + item.quantity * item.unit_price, 0)
    const taxAmount = subtotal * 0.1 // 10% tax
    const totalAmount = subtotal + taxAmount

    // Create purchase order
    const poResult = await query(
      `
      INSERT INTO purchase_order (
        sys_tenant_id, po_number, supplier_id, warehouse_id, po_date, 
        expected_delivery_date, subtotal, tax_amount, total_amount, notes, created_by
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
      RETURNING po_id
    `,
      [
        sys_tenant_id,
        poNumber,
        supplier_id,
        warehouse_id,
        po_date,
        expected_delivery_date,
        subtotal,
        taxAmount,
        totalAmount,
        notes,
        created_by,
      ],
    )

    const poId = poResult.rows[0].po_id

    // Create line items
    for (const item of line_items) {
      await query(
        `
        INSERT INTO purchase_order_line (
          sys_tenant_id, po_id, variant_id, quantity, unit_price
        ) VALUES ($1, $2, $3, $4, $5)
      `,
        [sys_tenant_id, poId, item.variant_id, item.quantity, item.unit_price],
      )
    }

    return NextResponse.json({
      success: true,
      po_id: poId,
      po_number: poNumber,
    })
  } catch (error) {
    console.error("Create Purchase Order error:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}
