import { type NextRequest, NextResponse } from "next/server"
import { neon } from "@neondatabase/serverless"

const sql = neon(process.env.DATABASE_URL!)

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const reportType = searchParams.get("type")
    const startDate = searchParams.get("startDate")
    const endDate = searchParams.get("endDate")
    const tenantId = 1 // Get from auth context

    let reportData

    switch (reportType) {
      case "balance-sheet":
        reportData = await generateBalanceSheet(tenantId, endDate || new Date().toISOString().split("T")[0])
        break
      case "income-statement":
        reportData = await generateIncomeStatement(tenantId, startDate!, endDate!)
        break
      case "trial-balance":
        reportData = await generateTrialBalance(tenantId, endDate || new Date().toISOString().split("T")[0])
        break
      default:
        return NextResponse.json({ error: "Invalid report type" }, { status: 400 })
    }

    return NextResponse.json(reportData)
  } catch (error) {
    console.error("Error generating financial report:", error)
    return NextResponse.json({ error: "Failed to generate report" }, { status: 500 })
  }
}

async function generateBalanceSheet(tenantId: number, asOfDate: string) {
  const accounts = await sql`
    SELECT 
      coa.account_code,
      coa.account_name,
      coa.account_type,
      COALESCE(SUM(jel.debit_amount - jel.credit_amount), 0) as balance
    FROM chart_of_accounts coa
    LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
    LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id
    WHERE coa.sys_tenant_id = ${tenantId}
      AND coa.account_type IN ('Asset', 'Liability', 'Equity')
      AND (je.entry_date IS NULL OR je.entry_date <= ${asOfDate})
      AND (je.status IS NULL OR je.status = 'posted')
    GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type
    ORDER BY coa.account_code
  `

  const assets = accounts.filter((acc) => acc.account_type === "Asset")
  const liabilities = accounts.filter((acc) => acc.account_type === "Liability")
  const equity = accounts.filter((acc) => acc.account_type === "Equity")

  return {
    reportType: "Balance Sheet",
    asOfDate,
    assets,
    liabilities,
    equity,
    totalAssets: assets.reduce((sum, acc) => sum + Number.parseFloat(acc.balance), 0),
    totalLiabilities: liabilities.reduce((sum, acc) => sum + Number.parseFloat(acc.balance), 0),
    totalEquity: equity.reduce((sum, acc) => sum + Number.parseFloat(acc.balance), 0),
  }
}

async function generateIncomeStatement(tenantId: number, startDate: string, endDate: string) {
  const accounts = await sql`
    SELECT 
      coa.account_code,
      coa.account_name,
      coa.account_type,
      COALESCE(SUM(jel.credit_amount - jel.debit_amount), 0) as balance
    FROM chart_of_accounts coa
    LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
    LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id
    WHERE coa.sys_tenant_id = ${tenantId}
      AND coa.account_type IN ('Revenue', 'Expense')
      AND je.entry_date BETWEEN ${startDate} AND ${endDate}
      AND je.status = 'posted'
    GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type
    ORDER BY coa.account_code
  `

  const revenue = accounts.filter((acc) => acc.account_type === "Revenue")
  const expenses = accounts.filter((acc) => acc.account_type === "Expense")

  const totalRevenue = revenue.reduce((sum, acc) => sum + Number.parseFloat(acc.balance), 0)
  const totalExpenses = expenses.reduce((sum, acc) => sum + Math.abs(Number.parseFloat(acc.balance)), 0)

  return {
    reportType: "Income Statement",
    period: `${startDate} to ${endDate}`,
    revenue,
    expenses,
    totalRevenue,
    totalExpenses,
    netIncome: totalRevenue - totalExpenses,
  }
}

async function generateTrialBalance(tenantId: number, asOfDate: string) {
  const accounts = await sql`
    SELECT 
      coa.account_code,
      coa.account_name,
      coa.account_type,
      COALESCE(SUM(jel.debit_amount), 0) as total_debits,
      COALESCE(SUM(jel.credit_amount), 0) as total_credits,
      COALESCE(SUM(jel.debit_amount - jel.credit_amount), 0) as balance
    FROM chart_of_accounts coa
    LEFT JOIN journal_entry_lines jel ON coa.id = jel.account_id
    LEFT JOIN journal_entries je ON jel.journal_entry_id = je.id
    WHERE coa.sys_tenant_id = ${tenantId}
      AND (je.entry_date IS NULL OR je.entry_date <= ${asOfDate})
      AND (je.status IS NULL OR je.status = 'posted')
    GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type
    HAVING COALESCE(SUM(jel.debit_amount), 0) != 0 OR COALESCE(SUM(jel.credit_amount), 0) != 0
    ORDER BY coa.account_code
  `

  const totalDebits = accounts.reduce((sum, acc) => sum + Number.parseFloat(acc.total_debits), 0)
  const totalCredits = accounts.reduce((sum, acc) => sum + Number.parseFloat(acc.total_credits), 0)

  return {
    reportType: "Trial Balance",
    asOfDate,
    accounts,
    totalDebits,
    totalCredits,
    isBalanced: Math.abs(totalDebits - totalCredits) < 0.01,
  }
}
