import { type NextRequest, NextResponse } from "next/server"
import { query } from "@/lib/db"

export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url)
    const tenantId = searchParams.get("tenant_id")

    if (!tenantId) {
      return NextResponse.json({ error: "Tenant ID is required" }, { status: 400 })
    }

    const result = await query(
      `
      SELECT 
        so.so_id,
        so.so_number,
        so.order_date,
        so.delivery_date,
        so.status,
        so.total_amount,
        so.created_at,
        c.customer_name,
        u.first_name || ' ' || u.last_name as created_by_name
      FROM sales_order so
      JOIN customer c ON so.customer_id = c.customer_id
      JOIN sys_user u ON so.created_by = u.user_id
      WHERE so.sys_tenant_id = $1
      ORDER BY so.created_at DESC
    `,
      [tenantId],
    )

    return NextResponse.json({
      salesOrders: result.rows,
    })
  } catch (error) {
    console.error("Sales Orders API error:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json()
    const { customer_id, warehouse_id, order_date, delivery_date, line_items, notes, created_by, sys_tenant_id } = body

    // Generate SO number
    const soNumberResult = await query(
      `
      SELECT 'SO-' || TO_CHAR(CURRENT_DATE, 'YYYY') || '-' || 
             LPAD((COALESCE(MAX(CAST(SUBSTRING(so_number FROM 9) AS INTEGER)), 0) + 1)::TEXT, 3, '0') as so_number
      FROM sales_order 
      WHERE sys_tenant_id = $1 AND so_number LIKE 'SO-' || TO_CHAR(CURRENT_DATE, 'YYYY') || '-%'
    `,
      [sys_tenant_id],
    )

    const soNumber = soNumberResult.rows[0].so_number

    // Calculate totals
    const subtotal = line_items.reduce((sum: number, item: any) => sum + item.quantity * item.unit_price, 0)
    const taxAmount = subtotal * 0.1 // 10% tax
    const totalAmount = subtotal + taxAmount

    // Create sales order
    const soResult = await query(
      `
      INSERT INTO sales_order (
        sys_tenant_id, so_number, customer_id, warehouse_id, order_date, 
        delivery_date, subtotal, tax_amount, total_amount, notes, created_by
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
      RETURNING so_id
    `,
      [
        sys_tenant_id,
        soNumber,
        customer_id,
        warehouse_id,
        order_date,
        delivery_date,
        subtotal,
        taxAmount,
        totalAmount,
        notes,
        created_by,
      ],
    )

    const soId = soResult.rows[0].so_id

    // Create line items
    for (const item of line_items) {
      await query(
        `
        INSERT INTO sales_order_line (
          sys_tenant_id, so_id, variant_id, quantity, unit_price
        ) VALUES ($1, $2, $3, $4, $5)
      `,
        [sys_tenant_id, soId, item.variant_id, item.quantity, item.unit_price],
      )
    }

    return NextResponse.json({
      success: true,
      so_id: soId,
      so_number: soNumber,
    })
  } catch (error) {
    console.error("Create Sales Order error:", error)
    return NextResponse.json({ error: "Internal server error" }, { status: 500 })
  }
}
