"use client"

import { useEffect, useState } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Edit, Eye, Search, FileText, DollarSign } from "lucide-react"

interface CustomerInvoice {
  invoice_id: string
  invoice_number: string
  customer_name: string
  dn_number: string
  invoice_date: string
  due_date: string
  status: string
  total_amount: number
  balance_due: number
  created_by_name: string
  created_at: string
}

interface CustomerInvoicesTableProps {
  tenantId: string
}

const statusColors = {
  Draft: "secondary",
  Posted: "default",
  Paid: "default",
  Overdue: "destructive",
  Cancelled: "destructive",
} as const

export function CustomerInvoicesTable({ tenantId }: CustomerInvoicesTableProps) {
  const [customerInvoices, setCustomerInvoices] = useState<CustomerInvoice[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState("")

  useEffect(() => {
    fetchCustomerInvoices()
  }, [tenantId])

  const fetchCustomerInvoices = async () => {
    try {
      const response = await fetch(`/api/customer-invoices?tenant_id=${tenantId}`)
      if (response.ok) {
        const data = await response.json()
        setCustomerInvoices(data.customerInvoices)
      }
    } catch (error) {
      console.error("Failed to fetch customer invoices:", error)
    } finally {
      setLoading(false)
    }
  }

  const filteredInvoices = customerInvoices.filter(
    (invoice) =>
      invoice.invoice_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
      invoice.customer_name.toLowerCase().includes(searchTerm.toLowerCase()),
  )

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Customer Invoices</CardTitle>
          <div className="relative w-64">
            <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search invoices..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-8"
            />
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Invoice Number</TableHead>
              <TableHead>Customer</TableHead>
              <TableHead>DN Number</TableHead>
              <TableHead>Invoice Date</TableHead>
              <TableHead>Due Date</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Total Amount</TableHead>
              <TableHead>Balance Due</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filteredInvoices.map((invoice) => (
              <TableRow key={invoice.invoice_id}>
                <TableCell className="font-medium">{invoice.invoice_number}</TableCell>
                <TableCell>{invoice.customer_name}</TableCell>
                <TableCell>{invoice.dn_number || "-"}</TableCell>
                <TableCell>{new Date(invoice.invoice_date).toLocaleDateString()}</TableCell>
                <TableCell>{new Date(invoice.due_date).toLocaleDateString()}</TableCell>
                <TableCell>
                  <Badge variant={statusColors[invoice.status as keyof typeof statusColors] || "secondary"}>
                    {invoice.status}
                  </Badge>
                </TableCell>
                <TableCell>${invoice.total_amount?.toLocaleString() || "0.00"}</TableCell>
                <TableCell>
                  <span className={invoice.balance_due > 0 ? "text-red-600 font-medium" : "text-green-600"}>
                    ${invoice.balance_due?.toLocaleString() || "0.00"}
                  </span>
                </TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    <Button variant="ghost" size="sm" title="View Details">
                      <Eye className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Edit">
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Record Payment">
                      <DollarSign className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Print">
                      <FileText className="h-4 w-4" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
        {filteredInvoices.length === 0 && (
          <div className="text-center py-8 text-muted-foreground">
            No customer invoices found. Create your first invoice to get started.
          </div>
        )}
      </CardContent>
    </Card>
  )
}
