"use client"

import { useEffect, useState } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Edit, Eye, Search, FileText } from "lucide-react"

interface DeliveryNote {
  dn_id: string
  dn_number: string
  so_number: string
  customer_name: string
  warehouse_name: string
  delivery_date: string
  status: string
  created_by_name: string
  created_at: string
}

interface DeliveryNotesTableProps {
  tenantId: string
}

const statusColors = {
  Draft: "secondary",
  Posted: "default",
  Cancelled: "destructive",
} as const

export function DeliveryNotesTable({ tenantId }: DeliveryNotesTableProps) {
  const [deliveryNotes, setDeliveryNotes] = useState<DeliveryNote[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState("")

  useEffect(() => {
    fetchDeliveryNotes()
  }, [tenantId])

  const fetchDeliveryNotes = async () => {
    try {
      const response = await fetch(`/api/delivery-notes?tenant_id=${tenantId}`)
      if (response.ok) {
        const data = await response.json()
        setDeliveryNotes(data.deliveryNotes)
      }
    } catch (error) {
      console.error("Failed to fetch delivery notes:", error)
    } finally {
      setLoading(false)
    }
  }

  const filteredNotes = deliveryNotes.filter(
    (note) =>
      note.dn_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
      note.so_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
      note.customer_name.toLowerCase().includes(searchTerm.toLowerCase()),
  )

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Delivery Notes</CardTitle>
          <div className="relative w-64">
            <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search delivery notes..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-8"
            />
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>DN Number</TableHead>
              <TableHead>SO Number</TableHead>
              <TableHead>Customer</TableHead>
              <TableHead>Warehouse</TableHead>
              <TableHead>Delivery Date</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Created By</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filteredNotes.map((note) => (
              <TableRow key={note.dn_id}>
                <TableCell className="font-medium">{note.dn_number}</TableCell>
                <TableCell>{note.so_number}</TableCell>
                <TableCell>{note.customer_name}</TableCell>
                <TableCell>{note.warehouse_name}</TableCell>
                <TableCell>{new Date(note.delivery_date).toLocaleDateString()}</TableCell>
                <TableCell>
                  <Badge variant={statusColors[note.status as keyof typeof statusColors] || "secondary"}>
                    {note.status}
                  </Badge>
                </TableCell>
                <TableCell>{note.created_by_name}</TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    <Button variant="ghost" size="sm" title="View Details">
                      <Eye className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Edit">
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Print">
                      <FileText className="h-4 w-4" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
        {filteredNotes.length === 0 && (
          <div className="text-center py-8 text-muted-foreground">
            No delivery notes found. Create your first delivery note to get started.
          </div>
        )}
      </CardContent>
    </Card>
  )
}
