"use client"

import { useEffect, useState } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Edit, Eye, Search, FileText } from "lucide-react"

interface GoodsReceipt {
  grn_id: string
  grn_number: string
  po_number: string
  supplier_name: string
  warehouse_name: string
  receipt_date: string
  status: string
  created_by_name: string
  created_at: string
}

interface GoodsReceiptTableProps {
  tenantId: string
}

const statusColors = {
  Draft: "secondary",
  Posted: "default",
  Cancelled: "destructive",
} as const

export function GoodsReceiptTable({ tenantId }: GoodsReceiptTableProps) {
  const [goodsReceipts, setGoodsReceipts] = useState<GoodsReceipt[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState("")

  useEffect(() => {
    fetchGoodsReceipts()
  }, [tenantId])

  const fetchGoodsReceipts = async () => {
    try {
      const response = await fetch(`/api/goods-receipt?tenant_id=${tenantId}`)
      if (response.ok) {
        const data = await response.json()
        setGoodsReceipts(data.goodsReceipts)
      }
    } catch (error) {
      console.error("Failed to fetch goods receipts:", error)
    } finally {
      setLoading(false)
    }
  }

  const filteredReceipts = goodsReceipts.filter(
    (receipt) =>
      receipt.grn_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
      receipt.po_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
      receipt.supplier_name.toLowerCase().includes(searchTerm.toLowerCase()),
  )

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Goods Receipt Notes</CardTitle>
          <div className="relative w-64">
            <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search GRNs..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-8"
            />
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>GRN Number</TableHead>
              <TableHead>PO Number</TableHead>
              <TableHead>Supplier</TableHead>
              <TableHead>Warehouse</TableHead>
              <TableHead>Receipt Date</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Created By</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filteredReceipts.map((receipt) => (
              <TableRow key={receipt.grn_id}>
                <TableCell className="font-medium">{receipt.grn_number}</TableCell>
                <TableCell>{receipt.po_number}</TableCell>
                <TableCell>{receipt.supplier_name}</TableCell>
                <TableCell>{receipt.warehouse_name}</TableCell>
                <TableCell>{new Date(receipt.receipt_date).toLocaleDateString()}</TableCell>
                <TableCell>
                  <Badge variant={statusColors[receipt.status as keyof typeof statusColors] || "secondary"}>
                    {receipt.status}
                  </Badge>
                </TableCell>
                <TableCell>{receipt.created_by_name}</TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    <Button variant="ghost" size="sm" title="View Details">
                      <Eye className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Edit">
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Print">
                      <FileText className="h-4 w-4" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
        {filteredReceipts.length === 0 && (
          <div className="text-center py-8 text-muted-foreground">
            No goods receipt notes found. Create your first GRN to get started.
          </div>
        )}
      </CardContent>
    </Card>
  )
}
