"use client"

import { useEffect, useState } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Edit, Search, AlertTriangle, Package } from "lucide-react"

interface InventoryItem {
  stock_id: string
  product_code: string
  product_name: string
  variant_code: string
  variant_name: string
  warehouse_name: string
  quantity_on_hand: number
  quantity_reserved: number
  quantity_available: number
  reorder_level: number
  cost_price: number
  selling_price: number
  stock_value: number
  is_low_stock: boolean
}

interface InventoryTableProps {
  tenantId: string
}

export function InventoryTable({ tenantId }: InventoryTableProps) {
  const [inventory, setInventory] = useState<InventoryItem[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState("")
  const [warehouseFilter, setWarehouseFilter] = useState("all")
  const [stockFilter, setStockFilter] = useState("all")
  const [warehouses, setWarehouses] = useState<{ warehouse_id: string; warehouse_name: string }[]>([])

  useEffect(() => {
    fetchInventory()
    fetchWarehouses()
  }, [tenantId])

  const fetchInventory = async () => {
    try {
      const response = await fetch(`/api/inventory?tenant_id=${tenantId}`)
      if (response.ok) {
        const data = await response.json()
        setInventory(data.inventory)
      }
    } catch (error) {
      console.error("Failed to fetch inventory:", error)
    } finally {
      setLoading(false)
    }
  }

  const fetchWarehouses = async () => {
    try {
      const response = await fetch(`/api/warehouses?tenant_id=${tenantId}`)
      if (response.ok) {
        const data = await response.json()
        setWarehouses(data.warehouses)
      }
    } catch (error) {
      console.error("Failed to fetch warehouses:", error)
    }
  }

  const filteredInventory = inventory.filter((item) => {
    const matchesSearch =
      item.product_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      item.product_code.toLowerCase().includes(searchTerm.toLowerCase()) ||
      item.variant_code.toLowerCase().includes(searchTerm.toLowerCase())

    const matchesWarehouse = warehouseFilter === "all" || item.warehouse_name === warehouseFilter

    const matchesStock =
      stockFilter === "all" ||
      (stockFilter === "low" && item.is_low_stock) ||
      (stockFilter === "available" && item.quantity_available > 0) ||
      (stockFilter === "out" && item.quantity_available <= 0)

    return matchesSearch && matchesWarehouse && matchesStock
  })

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Stock Levels</CardTitle>
          <div className="flex items-center gap-4">
            <div className="relative w-64">
              <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="Search inventory..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-8"
              />
            </div>
            <Select value={warehouseFilter} onValueChange={setWarehouseFilter}>
              <SelectTrigger className="w-40">
                <SelectValue placeholder="Warehouse" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Warehouses</SelectItem>
                {warehouses.map((warehouse) => (
                  <SelectItem key={warehouse.warehouse_id} value={warehouse.warehouse_name}>
                    {warehouse.warehouse_name}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            <Select value={stockFilter} onValueChange={setStockFilter}>
              <SelectTrigger className="w-32">
                <SelectValue placeholder="Stock Status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Items</SelectItem>
                <SelectItem value="low">Low Stock</SelectItem>
                <SelectItem value="available">Available</SelectItem>
                <SelectItem value="out">Out of Stock</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Product</TableHead>
              <TableHead>Variant</TableHead>
              <TableHead>Warehouse</TableHead>
              <TableHead>On Hand</TableHead>
              <TableHead>Reserved</TableHead>
              <TableHead>Available</TableHead>
              <TableHead>Reorder Level</TableHead>
              <TableHead>Stock Value</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filteredInventory.map((item) => (
              <TableRow key={item.stock_id}>
                <TableCell>
                  <div>
                    <div className="font-medium">{item.product_name}</div>
                    <div className="text-sm text-muted-foreground">{item.product_code}</div>
                  </div>
                </TableCell>
                <TableCell>
                  <div>
                    <div className="font-medium">{item.variant_name}</div>
                    <div className="text-sm text-muted-foreground">{item.variant_code}</div>
                  </div>
                </TableCell>
                <TableCell>{item.warehouse_name}</TableCell>
                <TableCell className="font-medium">{item.quantity_on_hand}</TableCell>
                <TableCell>{item.quantity_reserved}</TableCell>
                <TableCell className="font-medium">{item.quantity_available}</TableCell>
                <TableCell>{item.reorder_level}</TableCell>
                <TableCell>${item.stock_value?.toLocaleString() || "0.00"}</TableCell>
                <TableCell>
                  {item.is_low_stock ? (
                    <Badge variant="destructive" className="flex items-center gap-1 w-fit">
                      <AlertTriangle className="h-3 w-3" />
                      Low Stock
                    </Badge>
                  ) : item.quantity_available > 0 ? (
                    <Badge variant="default" className="flex items-center gap-1 w-fit">
                      <Package className="h-3 w-3" />
                      In Stock
                    </Badge>
                  ) : (
                    <Badge variant="secondary">Out of Stock</Badge>
                  )}
                </TableCell>
                <TableCell>
                  <Button variant="ghost" size="sm" title="Adjust Stock">
                    <Edit className="h-4 w-4" />
                  </Button>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
        {filteredInventory.length === 0 && (
          <div className="text-center py-8 text-muted-foreground">No inventory items found matching your criteria.</div>
        )}
      </CardContent>
    </Card>
  )
}
