"use client"

import { useEffect, useState } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Edit, Eye, Search, FileText } from "lucide-react"

interface JournalEntry {
  entry_id: string
  entry_number: string
  entry_date: string
  reference_type: string
  reference_id: string
  description: string
  total_debit: number
  total_credit: number
  status: string
  created_by_name: string
  created_at: string
}

interface JournalEntriesTableProps {
  tenantId: string
}

const statusColors = {
  Draft: "secondary",
  Posted: "default",
  Reversed: "destructive",
} as const

const referenceTypeColors = {
  PO: "default",
  GRN: "default",
  SI: "destructive",
  SO: "default",
  DN: "default",
  CI: "default",
  Manual: "secondary",
} as const

export function JournalEntriesTable({ tenantId }: JournalEntriesTableProps) {
  const [journalEntries, setJournalEntries] = useState<JournalEntry[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState("")
  const [statusFilter, setStatusFilter] = useState("all")
  const [typeFilter, setTypeFilter] = useState("all")

  useEffect(() => {
    fetchJournalEntries()
  }, [tenantId])

  const fetchJournalEntries = async () => {
    try {
      const response = await fetch(`/api/journal-entries?tenant_id=${tenantId}`)
      if (response.ok) {
        const data = await response.json()
        setJournalEntries(data.journalEntries)
      }
    } catch (error) {
      console.error("Failed to fetch journal entries:", error)
    } finally {
      setLoading(false)
    }
  }

  const filteredEntries = journalEntries.filter((entry) => {
    const matchesSearch =
      entry.entry_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
      entry.description.toLowerCase().includes(searchTerm.toLowerCase())

    const matchesStatus = statusFilter === "all" || entry.status === statusFilter
    const matchesType = typeFilter === "all" || entry.reference_type === typeFilter

    return matchesSearch && matchesStatus && matchesType
  })

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Journal Entries</CardTitle>
          <div className="flex items-center gap-4">
            <div className="relative w-64">
              <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="Search entries..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-8"
              />
            </div>
            <Select value={statusFilter} onValueChange={setStatusFilter}>
              <SelectTrigger className="w-32">
                <SelectValue placeholder="Status" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Status</SelectItem>
                <SelectItem value="Draft">Draft</SelectItem>
                <SelectItem value="Posted">Posted</SelectItem>
                <SelectItem value="Reversed">Reversed</SelectItem>
              </SelectContent>
            </Select>
            <Select value={typeFilter} onValueChange={setTypeFilter}>
              <SelectTrigger className="w-32">
                <SelectValue placeholder="Type" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Types</SelectItem>
                <SelectItem value="PO">Purchase Order</SelectItem>
                <SelectItem value="GRN">Goods Receipt</SelectItem>
                <SelectItem value="SI">Supplier Invoice</SelectItem>
                <SelectItem value="SO">Sales Order</SelectItem>
                <SelectItem value="DN">Delivery Note</SelectItem>
                <SelectItem value="CI">Customer Invoice</SelectItem>
                <SelectItem value="Manual">Manual Entry</SelectItem>
              </SelectContent>
            </Select>
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Entry Number</TableHead>
              <TableHead>Date</TableHead>
              <TableHead>Reference</TableHead>
              <TableHead>Description</TableHead>
              <TableHead>Debit</TableHead>
              <TableHead>Credit</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Created By</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filteredEntries.map((entry) => (
              <TableRow key={entry.entry_id}>
                <TableCell className="font-medium">{entry.entry_number}</TableCell>
                <TableCell>{new Date(entry.entry_date).toLocaleDateString()}</TableCell>
                <TableCell>
                  <Badge
                    variant={
                      referenceTypeColors[entry.reference_type as keyof typeof referenceTypeColors] || "secondary"
                    }
                  >
                    {entry.reference_type}
                  </Badge>
                </TableCell>
                <TableCell className="max-w-xs truncate">{entry.description}</TableCell>
                <TableCell className="font-medium">${entry.total_debit?.toLocaleString() || "0.00"}</TableCell>
                <TableCell className="font-medium">${entry.total_credit?.toLocaleString() || "0.00"}</TableCell>
                <TableCell>
                  <Badge variant={statusColors[entry.status as keyof typeof statusColors] || "secondary"}>
                    {entry.status}
                  </Badge>
                </TableCell>
                <TableCell>{entry.created_by_name}</TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    <Button variant="ghost" size="sm" title="View Details">
                      <Eye className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Edit">
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Print">
                      <FileText className="h-4 w-4" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
        {filteredEntries.length === 0 && (
          <div className="text-center py-8 text-muted-foreground">No journal entries found matching your criteria.</div>
        )}
      </CardContent>
    </Card>
  )
}
