"use client"

import { useEffect, useState } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Edit, Eye, Search, FileText, Truck } from "lucide-react"

interface PurchaseOrder {
  po_id: string
  po_number: string
  supplier_name: string
  po_date: string
  expected_delivery_date: string
  status: string
  total_amount: number
  created_by_name: string
  created_at: string
}

interface PurchaseOrdersTableProps {
  tenantId: string
}

const statusColors = {
  Draft: "secondary",
  Sent: "default",
  Confirmed: "default",
  "Partially Received": "default",
  "Fully Received": "default",
  Cancelled: "destructive",
} as const

export function PurchaseOrdersTable({ tenantId }: PurchaseOrdersTableProps) {
  const [purchaseOrders, setPurchaseOrders] = useState<PurchaseOrder[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState("")

  useEffect(() => {
    fetchPurchaseOrders()
  }, [tenantId])

  const fetchPurchaseOrders = async () => {
    try {
      const response = await fetch(`/api/purchase-orders?tenant_id=${tenantId}`)
      if (response.ok) {
        const data = await response.json()
        setPurchaseOrders(data.purchaseOrders)
      }
    } catch (error) {
      console.error("Failed to fetch purchase orders:", error)
    } finally {
      setLoading(false)
    }
  }

  const filteredOrders = purchaseOrders.filter(
    (order) =>
      order.po_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
      order.supplier_name.toLowerCase().includes(searchTerm.toLowerCase()),
  )

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Purchase Orders</CardTitle>
          <div className="relative w-64">
            <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search purchase orders..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-8"
            />
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>PO Number</TableHead>
              <TableHead>Supplier</TableHead>
              <TableHead>PO Date</TableHead>
              <TableHead>Expected Delivery</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Total Amount</TableHead>
              <TableHead>Created By</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filteredOrders.map((order) => (
              <TableRow key={order.po_id}>
                <TableCell className="font-medium">{order.po_number}</TableCell>
                <TableCell>{order.supplier_name}</TableCell>
                <TableCell>{new Date(order.po_date).toLocaleDateString()}</TableCell>
                <TableCell>
                  {order.expected_delivery_date ? new Date(order.expected_delivery_date).toLocaleDateString() : "-"}
                </TableCell>
                <TableCell>
                  <Badge variant={statusColors[order.status as keyof typeof statusColors] || "secondary"}>
                    {order.status}
                  </Badge>
                </TableCell>
                <TableCell>${order.total_amount?.toLocaleString() || "0.00"}</TableCell>
                <TableCell>{order.created_by_name}</TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    <Button variant="ghost" size="sm" title="View Details">
                      <Eye className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Edit">
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Generate GRN">
                      <Truck className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Print">
                      <FileText className="h-4 w-4" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
        {filteredOrders.length === 0 && (
          <div className="text-center py-8 text-muted-foreground">
            No purchase orders found. Create your first purchase order to get started.
          </div>
        )}
      </CardContent>
    </Card>
  )
}
