"use client"

import { useEffect, useState } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Edit, Eye, Search, FileText, Truck } from "lucide-react"

interface SalesOrder {
  so_id: string
  so_number: string
  customer_name: string
  order_date: string
  delivery_date: string
  status: string
  total_amount: number
  created_by_name: string
  created_at: string
}

interface SalesOrdersTableProps {
  tenantId: string
}

const statusColors = {
  Draft: "secondary",
  Confirmed: "default",
  "Partially Delivered": "default",
  "Fully Delivered": "default",
  Invoiced: "default",
  Cancelled: "destructive",
} as const

export function SalesOrdersTable({ tenantId }: SalesOrdersTableProps) {
  const [salesOrders, setSalesOrders] = useState<SalesOrder[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState("")

  useEffect(() => {
    fetchSalesOrders()
  }, [tenantId])

  const fetchSalesOrders = async () => {
    try {
      const response = await fetch(`/api/sales-orders?tenant_id=${tenantId}`)
      if (response.ok) {
        const data = await response.json()
        setSalesOrders(data.salesOrders)
      }
    } catch (error) {
      console.error("Failed to fetch sales orders:", error)
    } finally {
      setLoading(false)
    }
  }

  const filteredOrders = salesOrders.filter(
    (order) =>
      order.so_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
      order.customer_name.toLowerCase().includes(searchTerm.toLowerCase()),
  )

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Sales Orders</CardTitle>
          <div className="relative w-64">
            <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search sales orders..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-8"
            />
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>SO Number</TableHead>
              <TableHead>Customer</TableHead>
              <TableHead>Order Date</TableHead>
              <TableHead>Delivery Date</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Total Amount</TableHead>
              <TableHead>Created By</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filteredOrders.map((order) => (
              <TableRow key={order.so_id}>
                <TableCell className="font-medium">{order.so_number}</TableCell>
                <TableCell>{order.customer_name}</TableCell>
                <TableCell>{new Date(order.order_date).toLocaleDateString()}</TableCell>
                <TableCell>{order.delivery_date ? new Date(order.delivery_date).toLocaleDateString() : "-"}</TableCell>
                <TableCell>
                  <Badge variant={statusColors[order.status as keyof typeof statusColors] || "secondary"}>
                    {order.status}
                  </Badge>
                </TableCell>
                <TableCell>${order.total_amount?.toLocaleString() || "0.00"}</TableCell>
                <TableCell>{order.created_by_name}</TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    <Button variant="ghost" size="sm" title="View Details">
                      <Eye className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Edit">
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Create Delivery">
                      <Truck className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Print">
                      <FileText className="h-4 w-4" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
        {filteredOrders.length === 0 && (
          <div className="text-center py-8 text-muted-foreground">
            No sales orders found. Create your first sales order to get started.
          </div>
        )}
      </CardContent>
    </Card>
  )
}
