"use client"

import { useEffect, useState } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Edit, Eye, Search } from "lucide-react"

interface StockAdjustment {
  adjustment_id: string
  adjustment_number: string
  warehouse_name: string
  adjustment_date: string
  adjustment_type: string
  total_items: number
  status: string
  reason: string
  created_by_name: string
  created_at: string
}

interface StockAdjustmentsTableProps {
  tenantId: string
}

const statusColors = {
  Draft: "secondary",
  Posted: "default",
  Cancelled: "destructive",
} as const

const typeColors = {
  "Stock Loss": "destructive",
  "Stock Found": "default",
  Damage: "destructive",
  Correction: "secondary",
} as const

export function StockAdjustmentsTable({ tenantId }: StockAdjustmentsTableProps) {
  const [adjustments, setAdjustments] = useState<StockAdjustment[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState("")

  useEffect(() => {
    fetchAdjustments()
  }, [tenantId])

  const fetchAdjustments = async () => {
    try {
      const response = await fetch(`/api/stock-adjustments?tenant_id=${tenantId}`)
      if (response.ok) {
        const data = await response.json()
        setAdjustments(data.adjustments)
      }
    } catch (error) {
      console.error("Failed to fetch stock adjustments:", error)
    } finally {
      setLoading(false)
    }
  }

  const filteredAdjustments = adjustments.filter(
    (adjustment) =>
      adjustment.adjustment_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
      adjustment.warehouse_name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      adjustment.reason?.toLowerCase().includes(searchTerm.toLowerCase()),
  )

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Stock Adjustments</CardTitle>
          <div className="relative w-64">
            <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search adjustments..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-8"
            />
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Adjustment Number</TableHead>
              <TableHead>Warehouse</TableHead>
              <TableHead>Date</TableHead>
              <TableHead>Type</TableHead>
              <TableHead>Items</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Reason</TableHead>
              <TableHead>Created By</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filteredAdjustments.map((adjustment) => (
              <TableRow key={adjustment.adjustment_id}>
                <TableCell className="font-medium">{adjustment.adjustment_number}</TableCell>
                <TableCell>{adjustment.warehouse_name}</TableCell>
                <TableCell>{new Date(adjustment.adjustment_date).toLocaleDateString()}</TableCell>
                <TableCell>
                  <Badge variant={typeColors[adjustment.adjustment_type as keyof typeof typeColors] || "secondary"}>
                    {adjustment.adjustment_type}
                  </Badge>
                </TableCell>
                <TableCell>{adjustment.total_items}</TableCell>
                <TableCell>
                  <Badge variant={statusColors[adjustment.status as keyof typeof statusColors] || "secondary"}>
                    {adjustment.status}
                  </Badge>
                </TableCell>
                <TableCell className="max-w-xs truncate">{adjustment.reason || "-"}</TableCell>
                <TableCell>{adjustment.created_by_name}</TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    <Button variant="ghost" size="sm" title="View Details">
                      <Eye className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Edit">
                      <Edit className="h-4 w-4" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
        {filteredAdjustments.length === 0 && (
          <div className="text-center py-8 text-muted-foreground">
            No stock adjustments found. Create your first adjustment to get started.
          </div>
        )}
      </CardContent>
    </Card>
  )
}
