"use client"

import { useEffect, useState } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Edit, Eye, Search, FileText, CreditCard } from "lucide-react"

interface SupplierInvoice {
  invoice_id: string
  invoice_number: string
  supplier_invoice_number: string
  supplier_name: string
  grn_number: string
  invoice_date: string
  due_date: string
  status: string
  total_amount: number
  balance_due: number
  created_by_name: string
  created_at: string
}

interface SupplierInvoicesTableProps {
  tenantId: string
}

const statusColors = {
  Draft: "secondary",
  Posted: "default",
  Paid: "default",
  Cancelled: "destructive",
} as const

export function SupplierInvoicesTable({ tenantId }: SupplierInvoicesTableProps) {
  const [supplierInvoices, setSupplierInvoices] = useState<SupplierInvoice[]>([])
  const [loading, setLoading] = useState(true)
  const [searchTerm, setSearchTerm] = useState("")

  useEffect(() => {
    fetchSupplierInvoices()
  }, [tenantId])

  const fetchSupplierInvoices = async () => {
    try {
      const response = await fetch(`/api/supplier-invoices?tenant_id=${tenantId}`)
      if (response.ok) {
        const data = await response.json()
        setSupplierInvoices(data.supplierInvoices)
      }
    } catch (error) {
      console.error("Failed to fetch supplier invoices:", error)
    } finally {
      setLoading(false)
    }
  }

  const filteredInvoices = supplierInvoices.filter(
    (invoice) =>
      invoice.invoice_number.toLowerCase().includes(searchTerm.toLowerCase()) ||
      invoice.supplier_invoice_number?.toLowerCase().includes(searchTerm.toLowerCase()) ||
      invoice.supplier_name.toLowerCase().includes(searchTerm.toLowerCase()),
  )

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
          </div>
        </CardContent>
      </Card>
    )
  }

  return (
    <Card>
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>Supplier Invoices</CardTitle>
          <div className="relative w-64">
            <Search className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search invoices..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-8"
            />
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Invoice Number</TableHead>
              <TableHead>Supplier Invoice</TableHead>
              <TableHead>Supplier</TableHead>
              <TableHead>GRN</TableHead>
              <TableHead>Invoice Date</TableHead>
              <TableHead>Due Date</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Total Amount</TableHead>
              <TableHead>Balance Due</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {filteredInvoices.map((invoice) => (
              <TableRow key={invoice.invoice_id}>
                <TableCell className="font-medium">{invoice.invoice_number}</TableCell>
                <TableCell>{invoice.supplier_invoice_number || "-"}</TableCell>
                <TableCell>{invoice.supplier_name}</TableCell>
                <TableCell>{invoice.grn_number || "-"}</TableCell>
                <TableCell>{new Date(invoice.invoice_date).toLocaleDateString()}</TableCell>
                <TableCell>{new Date(invoice.due_date).toLocaleDateString()}</TableCell>
                <TableCell>
                  <Badge variant={statusColors[invoice.status as keyof typeof statusColors] || "secondary"}>
                    {invoice.status}
                  </Badge>
                </TableCell>
                <TableCell>${invoice.total_amount?.toLocaleString() || "0.00"}</TableCell>
                <TableCell>
                  <span className={invoice.balance_due > 0 ? "text-red-600 font-medium" : "text-green-600"}>
                    ${invoice.balance_due?.toLocaleString() || "0.00"}
                  </span>
                </TableCell>
                <TableCell>
                  <div className="flex items-center gap-2">
                    <Button variant="ghost" size="sm" title="View Details">
                      <Eye className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Edit">
                      <Edit className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Make Payment">
                      <CreditCard className="h-4 w-4" />
                    </Button>
                    <Button variant="ghost" size="sm" title="Print">
                      <FileText className="h-4 w-4" />
                    </Button>
                  </div>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
        {filteredInvoices.length === 0 && (
          <div className="text-center py-8 text-muted-foreground">
            No supplier invoices found. Create your first invoice to get started.
          </div>
        )}
      </CardContent>
    </Card>
  )
}
