import { neon } from "@neondatabase/serverless"

const sql = neon(process.env.DATABASE_URL!)

export interface JournalEntry {
  id?: number
  sys_tenant_id: number
  entry_number: string
  entry_date: string
  description: string
  reference_type?: string
  reference_id?: number
  total_debit: number
  total_credit: number
  status: "draft" | "posted" | "reversed"
  created_by?: number
  lines: JournalEntryLine[]
}

export interface JournalEntryLine {
  id?: number
  account_id: number
  account_code?: string
  account_name?: string
  description: string
  debit_amount: number
  credit_amount: number
}

export async function createJournalEntry(entry: JournalEntry): Promise<number> {
  const [journalEntry] = await sql`
    INSERT INTO journal_entries (
      sys_tenant_id, entry_number, entry_date, description, 
      reference_type, reference_id, total_debit, total_credit, 
      status, created_by
    ) VALUES (
      ${entry.sys_tenant_id}, ${entry.entry_number}, ${entry.entry_date}, 
      ${entry.description}, ${entry.reference_type}, ${entry.reference_id}, 
      ${entry.total_debit}, ${entry.total_credit}, ${entry.status}, ${entry.created_by}
    ) RETURNING id
  `

  // Insert journal entry lines
  for (const line of entry.lines) {
    await sql`
      INSERT INTO journal_entry_lines (
        sys_tenant_id, journal_entry_id, account_id, description, 
        debit_amount, credit_amount
      ) VALUES (
        ${entry.sys_tenant_id}, ${journalEntry.id}, ${line.account_id}, 
        ${line.description}, ${line.debit_amount}, ${line.credit_amount}
      )
    `
  }

  return journalEntry.id
}

// Automated journal entries for business transactions
export async function createPurchaseOrderJournalEntry(
  tenantId: number,
  poId: number,
  totalAmount: number,
  userId: number,
) {
  const entryNumber = `PO-${poId}-${Date.now()}`

  const entry: JournalEntry = {
    sys_tenant_id: tenantId,
    entry_number: entryNumber,
    entry_date: new Date().toISOString().split("T")[0],
    description: `Purchase Order #${poId}`,
    reference_type: "PO",
    reference_id: poId,
    total_debit: totalAmount,
    total_credit: totalAmount,
    status: "posted",
    created_by: userId,
    lines: [
      {
        account_id: 13, // Inventory account
        description: `Inventory from PO #${poId}`,
        debit_amount: totalAmount,
        credit_amount: 0,
      },
      {
        account_id: 10, // Accounts Payable
        description: `Accounts Payable for PO #${poId}`,
        debit_amount: 0,
        credit_amount: totalAmount,
      },
    ],
  }

  return await createJournalEntry(entry)
}

export async function createSalesOrderJournalEntry(
  tenantId: number,
  soId: number,
  totalAmount: number,
  userId: number,
) {
  const entryNumber = `SO-${soId}-${Date.now()}`

  const entry: JournalEntry = {
    sys_tenant_id: tenantId,
    entry_number: entryNumber,
    entry_date: new Date().toISOString().split("T")[0],
    description: `Sales Order #${soId}`,
    reference_type: "SO",
    reference_id: soId,
    total_debit: totalAmount,
    total_credit: totalAmount,
    status: "posted",
    created_by: userId,
    lines: [
      {
        account_id: 4, // Accounts Receivable
        description: `Accounts Receivable from SO #${soId}`,
        debit_amount: totalAmount,
        credit_amount: 0,
      },
      {
        account_id: 18, // Sales Revenue
        description: `Sales Revenue from SO #${soId}`,
        debit_amount: 0,
        credit_amount: totalAmount,
      },
    ],
  }

  return await createJournalEntry(entry)
}
