import { cookies } from "next/headers"
import { redirect } from "next/navigation"

export interface User {
  user_id: string
  sys_tenant_id: string
  username: string
  email: string
  first_name: string
  last_name: string
  roles: string[]
  permissions: string[]
}

export interface AuthSession {
  user: User
  token: string
}

const JWT_SECRET = process.env.JWT_SECRET || "your-secret-key-change-in-production"

export function generateToken(user: User): string {
  const payload = {
    user_id: user.user_id,
    sys_tenant_id: user.sys_tenant_id,
    username: user.username,
    email: user.email,
    first_name: user.first_name,
    last_name: user.last_name,
    roles: user.roles,
    permissions: user.permissions,
    exp: Date.now() + 24 * 60 * 60 * 1000, // 24 hours
  }
  return Buffer.from(JSON.stringify(payload)).toString("base64")
}

export function verifyToken(token: string): User | null {
  try {
    const decoded = JSON.parse(Buffer.from(token, "base64").toString())

    // Check if token is expired
    if (decoded.exp < Date.now()) {
      return null
    }

    return {
      user_id: decoded.user_id,
      sys_tenant_id: decoded.sys_tenant_id,
      username: decoded.username,
      email: decoded.email,
      first_name: decoded.first_name || "",
      last_name: decoded.last_name || "",
      roles: decoded.roles || [],
      permissions: decoded.permissions || [],
    }
  } catch {
    return null
  }
}

export async function getSession(): Promise<AuthSession | null> {
  const cookieStore = await cookies()
  const token = cookieStore.get("auth-token")?.value

  if (!token) return null

  const user = verifyToken(token)
  if (!user) return null

  return { user, token }
}

export async function requireAuth(): Promise<User> {
  const session = await getSession()
  if (!session) {
    redirect("/login")
  }
  return session.user
}

export function hasPermission(user: User, permission: string): boolean {
  return user.permissions.includes(permission) || user.roles.includes("Administrator")
}

export function requirePermission(user: User, permission: string): void {
  if (!hasPermission(user, permission)) {
    throw new Error(`Access denied. Required permission: ${permission}`)
  }
}
