-- Creating inventory management tables for stock adjustments and transfers

-- Stock Adjustments
CREATE TABLE IF NOT EXISTS stock_adjustment (
    adjustment_id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    sys_tenant_id UUID NOT NULL REFERENCES sys_tenant(tenant_id),
    adjustment_number VARCHAR(50) NOT NULL,
    warehouse_id UUID NOT NULL REFERENCES warehouse(warehouse_id),
    adjustment_date DATE NOT NULL,
    adjustment_type VARCHAR(50) NOT NULL CHECK (adjustment_type IN ('Stock Loss', 'Stock Found', 'Damage', 'Correction')),
    status VARCHAR(20) DEFAULT 'Draft' CHECK (status IN ('Draft', 'Posted', 'Cancelled')),
    reason TEXT,
    total_items INTEGER DEFAULT 0,
    created_by UUID NOT NULL REFERENCES sys_user(user_id),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(sys_tenant_id, adjustment_number)
);

-- Stock Adjustment Line Items
CREATE TABLE IF NOT EXISTS stock_adjustment_line (
    adjustment_line_id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    sys_tenant_id UUID NOT NULL REFERENCES sys_tenant(tenant_id),
    adjustment_id UUID NOT NULL REFERENCES stock_adjustment(adjustment_id) ON DELETE CASCADE,
    variant_id UUID NOT NULL REFERENCES product_variant(variant_id),
    quantity_adjusted DECIMAL(15,3) NOT NULL,
    unit_cost DECIMAL(15,2) NOT NULL,
    line_total DECIMAL(15,2) GENERATED ALWAYS AS (ABS(quantity_adjusted) * unit_cost) STORED,
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Stock Transfers
CREATE TABLE IF NOT EXISTS stock_transfer (
    transfer_id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    sys_tenant_id UUID NOT NULL REFERENCES sys_tenant(tenant_id),
    transfer_number VARCHAR(50) NOT NULL,
    from_warehouse_id UUID NOT NULL REFERENCES warehouse(warehouse_id),
    to_warehouse_id UUID NOT NULL REFERENCES warehouse(warehouse_id),
    transfer_date DATE NOT NULL,
    status VARCHAR(20) DEFAULT 'Draft' CHECK (status IN ('Draft', 'In Transit', 'Received', 'Cancelled')),
    notes TEXT,
    total_items INTEGER DEFAULT 0,
    created_by UUID NOT NULL REFERENCES sys_user(user_id),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(sys_tenant_id, transfer_number),
    CHECK (from_warehouse_id != to_warehouse_id)
);

-- Stock Transfer Line Items
CREATE TABLE IF NOT EXISTS stock_transfer_line (
    transfer_line_id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    sys_tenant_id UUID NOT NULL REFERENCES sys_tenant(tenant_id),
    transfer_id UUID NOT NULL REFERENCES stock_transfer(transfer_id) ON DELETE CASCADE,
    variant_id UUID NOT NULL REFERENCES product_variant(variant_id),
    quantity_transferred DECIMAL(15,3) NOT NULL,
    quantity_received DECIMAL(15,3) DEFAULT 0,
    unit_cost DECIMAL(15,2) NOT NULL,
    line_total DECIMAL(15,2) GENERATED ALWAYS AS (quantity_transferred * unit_cost) STORED,
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Stocktaking (Physical Inventory Count)
CREATE TABLE IF NOT EXISTS stocktaking (
    stocktaking_id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    sys_tenant_id UUID NOT NULL REFERENCES sys_tenant(tenant_id),
    stocktaking_number VARCHAR(50) NOT NULL,
    warehouse_id UUID NOT NULL REFERENCES warehouse(warehouse_id),
    count_date DATE NOT NULL,
    status VARCHAR(20) DEFAULT 'Draft' CHECK (status IN ('Draft', 'In Progress', 'Completed', 'Posted', 'Cancelled')),
    notes TEXT,
    total_items INTEGER DEFAULT 0,
    total_variance_value DECIMAL(15,2) DEFAULT 0,
    created_by UUID NOT NULL REFERENCES sys_user(user_id),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(sys_tenant_id, stocktaking_number)
);

-- Stocktaking Line Items
CREATE TABLE IF NOT EXISTS stocktaking_line (
    stocktaking_line_id UUID PRIMARY KEY DEFAULT gen_random_uuid(),
    sys_tenant_id UUID NOT NULL REFERENCES sys_tenant(tenant_id),
    stocktaking_id UUID NOT NULL REFERENCES stocktaking(stocktaking_id) ON DELETE CASCADE,
    variant_id UUID NOT NULL REFERENCES product_variant(variant_id),
    system_quantity DECIMAL(15,3) NOT NULL,
    counted_quantity DECIMAL(15,3),
    variance_quantity DECIMAL(15,3) GENERATED ALWAYS AS (COALESCE(counted_quantity, 0) - system_quantity) STORED,
    unit_cost DECIMAL(15,2) NOT NULL,
    variance_value DECIMAL(15,2) GENERATED ALWAYS AS ((COALESCE(counted_quantity, 0) - system_quantity) * unit_cost) STORED,
    notes TEXT,
    counted_by UUID REFERENCES sys_user(user_id),
    counted_at TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Create indexes for inventory tables
CREATE INDEX IF NOT EXISTS idx_stock_adjustment_tenant ON stock_adjustment(sys_tenant_id);
CREATE INDEX IF NOT EXISTS idx_stock_adjustment_warehouse ON stock_adjustment(warehouse_id);
CREATE INDEX IF NOT EXISTS idx_stock_transfer_tenant ON stock_transfer(sys_tenant_id);
CREATE INDEX IF NOT EXISTS idx_stock_transfer_from_warehouse ON stock_transfer(from_warehouse_id);
CREATE INDEX IF NOT EXISTS idx_stock_transfer_to_warehouse ON stock_transfer(to_warehouse_id);
CREATE INDEX IF NOT EXISTS idx_stocktaking_tenant ON stocktaking(sys_tenant_id);
CREATE INDEX IF NOT EXISTS idx_stocktaking_warehouse ON stocktaking(warehouse_id);
