// Content script to extract product information
function scrapeProductData() {
  // Get the current page URL
  const url = window.location.href;
  
  // Try to extract product information using common selectors
  const productData = {
    url: url,
    timestamp: new Date().toISOString(),
    title: '',
    price: '',
    currency: '',
    images: [],
    description: '',
    availability: '',
    rating: '',
    reviews: ''
  };
  
  // Try to get product title
  const titleSelectors = [
    'h1[data-testid="product-title"]',
    'h1.product-title',
    '.product-title',
    'h1.title',
    '.title',
    'h1'
  ];
  
  for (const selector of titleSelectors) {
    const element = document.querySelector(selector);
    if (element && element.textContent.trim()) {
      productData.title = element.textContent.trim();
      break;
    }
  }
  
  // If we still don't have a title, try the page title
  if (!productData.title) {
    productData.title = document.title;
  }
  
  // Try to get product price
  const priceSelectors = [
    '[data-testid="product-price"]',
    '.product-price',
    '.price-current',
    '.price',
    '[class*="price"]'
  ];
  
  for (const selector of priceSelectors) {
    const element = document.querySelector(selector);
    if (element && element.textContent.trim()) {
      const priceText = element.textContent.trim();
      productData.price = priceText;
      // Try to extract currency
      const currencyMatch = priceText.match(/([$£€¥₹]|USD|EUR|GBP|JPY|INR|CNY|CAD|AUD)/);
      if (currencyMatch) {
        productData.currency = currencyMatch[1];
      }
      break;
    }
  }
  
  // Try to get product images
  const imageSelectors = [
    '[data-testid="product-image"]',
    '.product-image img',
    '.product-img img',
    '.image img',
    'img[alt*="product" i]',
    '.main-image img',
    '#main-image img',
    'img'
  ];
  
  const images = [];
  for (const selector of imageSelectors) {
    const elements = document.querySelectorAll(selector);
    elements.forEach(img => {
      if (img.src && !img.src.includes('data:image')) {
        // Convert relative URLs to absolute URLs
        try {
          const absoluteUrl = new URL(img.src, window.location.origin);
          images.push(absoluteUrl.href);
        } catch (e) {
          // If URL parsing fails, just use the original src
          images.push(img.src);
        }
      }
    });
    
    if (images.length >= 3) { // Collect up to 3 images
      productData.images = [...new Set(images)].slice(0, 3); // Remove duplicates and limit to 3
      break;
    }
  }
  
  // If we still don't have images, try to get at least one
  if (productData.images.length === 0) {
    const allImages = document.querySelectorAll('img');
    for (const img of allImages) {
      if (img.src && !img.src.includes('data:image') && img.src.startsWith('http')) {
        try {
          const absoluteUrl = new URL(img.src, window.location.origin);
          productData.images.push(absoluteUrl.href);
        } catch (e) {
          productData.images.push(img.src);
        }
        // Just get the first valid image
        break;
      }
    }
  }
  
  // Try to get product description
  const descSelectors = [
    '[data-testid="product-description"]',
    '.product-description',
    '.description',
    '[class*="description"]',
    '#description',
    '[id*="description" i]'
  ];
  
  for (const selector of descSelectors) {
    const element = document.querySelector(selector);
    if (element && element.textContent.trim()) {
      productData.description = element.textContent.trim();
      break;
    }
  }
  
  // If we still don't have a description, try getting it from meta tags
  if (!productData.description) {
    const metaDescription = document.querySelector('meta[name="description"]');
    if (metaDescription && metaDescription.content) {
      productData.description = metaDescription.content;
    }
  }
  
  // Try to get availability
  const availabilitySelectors = [
    '[data-testid="availability"]',
    '.availability',
    '[class*="stock" i]',
    '[class*="availability" i]',
    '.stock'
  ];
  
  for (const selector of availabilitySelectors) {
    const element = document.querySelector(selector);
    if (element && element.textContent.trim()) {
      productData.availability = element.textContent.trim();
      break;
    }
  }
  
  // Try to get rating
  const ratingSelectors = [
    '[data-testid="rating"]',
    '.rating',
    '[class*="rating" i]',
    '[class*="review" i]',
    '.stars'
  ];
  
  for (const selector of ratingSelectors) {
    const element = document.querySelector(selector);
    if (element && element.textContent.trim()) {
      productData.rating = element.textContent.trim();
      break;
    }
  }
  
  // Try to get number of reviews
  const reviewSelectors = [
    '[data-testid="review-count"]',
    '.review-count',
    '[class*="review-count" i]',
    '[class*="reviews" i]',
    '.num-reviews'
  ];
  
  for (const selector of reviewSelectors) {
    const element = document.querySelector(selector);
    if (element && element.textContent.trim()) {
      productData.reviews = element.textContent.trim();
      break;
    }
  }
  
  return productData;
}

// Listen for messages from the popup or background script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === "scrapeProduct") {
    const productData = scrapeProductData();
    sendResponse({productData: productData});
  }
});