// =================================================================================
// --- 1. Constants & Helpers ---
// =================================================================================
const DB_NAME = 'ShopeeReviewDB';
const STORE_NAME = 'reviews';
const CAPTCHA_ALARM_NAME = 'captcha_timeout_alarm';

function openDB() {
    return new Promise((resolve, reject) => {
        const request = indexedDB.open(DB_NAME, 1);
        request.onerror = (event) => reject(`IndexedDB error: ${event.target.errorCode}`);
        request.onsuccess = (event) => resolve(event.target.result);
        request.onupgradeneeded = (event) => {
            const db = event.target.result;
            db.createObjectStore(STORE_NAME, { keyPath: 'review.cmtid' });
        };
    });
}

async function addDataToDB(data) {
    if (!data || data.length === 0) return Promise.resolve();
    const db = await openDB();
    return new Promise((resolve, reject) => {
        const transaction = db.transaction([STORE_NAME], 'readwrite');
        const store = transaction.objectStore(STORE_NAME);
        data.forEach(item => { if (item.review) store.put(item); });
        transaction.oncomplete = () => resolve();
        transaction.onerror = (event) => reject(`Transaction error: ${event.target.error}`);
    });
}

async function getAllDataFromDB() {
    const db = await openDB();
    return new Promise((resolve, reject) => {
        const transaction = db.transaction([STORE_NAME], 'readonly');
        const store = transaction.objectStore(STORE_NAME);
        const request = store.getAll();
        request.onsuccess = () => resolve(request.result);
        request.onerror = (event) => reject(`Error fetching all data: ${event.target.error}`);
    });
}

async function clearDB() {
    const db = await openDB();
    return new Promise((resolve, reject) => {
        const transaction = db.transaction([STORE_NAME], 'readwrite');
        const store = transaction.objectStore(STORE_NAME);
        const request = store.clear();
        request.onsuccess = () => resolve();
        request.onerror = (event) => reject(`Error clearing DB: ${event.target.error}`);
    });
}

function convertToFinalCSV(data) {
    const csvRows = [];
    const headers = [
        'product_name', 'shop_name', 'model_id', 'model_name', 'itemid', 'shopid', 'review_id', 'order_id',
        'author_username', 'anonymous', 'user_loyalty_tier', 'rating_star',
        'product_quality_rating', 'like_count', 'comment', 'review_tags',
        'review_creation_time', 'is_repeated_purchase', 'product_url',
        'media_url', 'description', 'media_type'
    ];
    csvRows.push(headers.map(h => `"${h}"`).join(';'));
    const add = (val) => {
        let str = String(val === null || val === undefined ? '' : val);
        // Sanitize newlines to prevent breaking CSV rows, then escape double quotes.
        str = str.replace(/(\r\n|\n|\r)/gm, " ");
        return `"${str.split('"').join('""')}"`;
    };
    data.forEach(item => {
        const review = item.review;
        if (review) {
            const ctime = review.ctime ? new Date(review.ctime * 1000) : null;
            const yymmdd = ctime ? ctime.toISOString().slice(2, 10).replace(/-/g, '') : '';

            const modelInfo = review.product_items && review.product_items[0] ? review.product_items[0] : {};
            const commonData = [
                add(item.productName), add(item.shopName), add(modelInfo.modelid), add(modelInfo.model_name),
                add(review.itemid), add(review.shopid), add(review.cmtid), add(review.orderid),
                add(review.author_username), add(review.anonymous), add(review.loyalty_info ? review.loyalty_info.tier_text : ''),
                add(review.rating_star), add(review.detailed_rating ? review.detailed_rating.product_quality : ''),
                add(review.like_count), add(review.comment), add(review.template_tags ? review.template_tags.join('|') : ''),
                add(ctime ? ctime.toISOString() : ''), add(review.is_repeated_purchase)
            ];

            const baseDataWithProductUrl = [...commonData, add(item.productUrl)];
            const description = `${yymmdd} *${review.rating_star} - ${review.author_username} - ${review.comment}`;
            const hasImages = review.images && review.images.length > 0;
            const hasVideos = review.videos && review.videos.length > 0;

            const productHostname = item.productUrl ? new URL(item.productUrl).hostname : 'shopee.co.id';
            const imageCdnUrl = `https://cf.${productHostname}/file/`;

            if (review.media_type === 'no_review') {
                csvRows.push([...baseDataWithProductUrl, add(''), add(description), add('no_review')].join(';'));
            } else if (!hasImages && !hasVideos) {
                csvRows.push([...baseDataWithProductUrl, add(''), add(description), add('text_only')].join(';'));
            } else {
                if (hasImages) review.images.forEach(image => csvRows.push([...baseDataWithProductUrl, add(`${imageCdnUrl}${image}`), add(description), add('image')].join(';')));
                if (hasVideos) review.videos.forEach(video => csvRows.push([...baseDataWithProductUrl, add(video.url), add(description), add('video')].join(';')));
            }
        }
    });
    return csvRows.join('\n');
}

// =================================================================================
// --- 3. Main Extension Logic ---
// =================================================================================

async function getTabId() {
    const result = await chrome.storage.session.get('work_tab_id');
    return result.work_tab_id;
}

async function setTabId(id) {
    await chrome.storage.session.set({ 'work_tab_id': id });
}

async function updateUrlStatus(url, status, summary) {
    const { urlStatuses } = await chrome.storage.local.get('urlStatuses');
    const statuses = urlStatuses || [];
    const index = statuses.findIndex(item => item.url === url || url.includes(item.url));
    if (index !== -1) {
        statuses[index].status = status;
        if (summary) statuses[index].summary = summary;
        await chrome.storage.local.set({ urlStatuses: statuses });
    }
}

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    const actions = {
        'start_sequence': () => startSequence(request.urls),
        'stop_sequence': () => finalizeSequence(true),
        'export_db_as_json': () => exportDbAsJson(),
        'scraped_reviews_for_db': () => addDataToDB(request.data),
        'url_processing_finished': async () => {
            await chrome.alarms.clear(CAPTCHA_ALARM_NAME);
            await chrome.storage.local.remove('pausedUrl');
            await updateUrlStatus(request.payload.url, request.payload.status, request.payload.summary);
            processNextUrl();
        },
        'pause_for_captcha': async () => {
            await updateUrlStatus(request.payload.url, 'captcha', request.payload.summary);
            await chrome.storage.local.set({
                processingStatus: 'PAUSED - Please solve CAPTCHA in scraping tab.',
                pausedUrl: request.payload.url
            });
            chrome.alarms.create(CAPTCHA_ALARM_NAME, { delayInMinutes: 5 });
        }
    };
    
    if (actions[request.action]) {
        actions[request.action]();
    }
    return true;
});

async function exportDbAsJson() {
    const allData = await getAllDataFromDB();
    if (allData && allData.length > 0) {
        const jsonString = JSON.stringify(allData, null, 2);
        const blob = new Blob([jsonString], { type: 'application/json' });
        const reader = new FileReader();
        reader.onload = function() {
            chrome.downloads.download({
                url: reader.result,
                filename: 'shopee_reviews_db_export.json',
                saveAs: false
            });
        };
        reader.readAsDataURL(blob);
    } else {
        console.log('[Bulk Scraper] No data in DB to export.');
    }
}

chrome.alarms.onAlarm.addListener(async (alarm) => {
    if (alarm.name === CAPTCHA_ALARM_NAME) {
        console.log('[Bulk Scraper] CAPTCHA alarm triggered. Resuming queue.');
        const { pausedUrl } = await chrome.storage.local.get('pausedUrl');
        if (pausedUrl) {
            await updateUrlStatus(pausedUrl, 'failed', 'Failed: CAPTCHA not solved within 5 minutes.');
            await chrome.storage.local.remove('pausedUrl');
            processNextUrl();
        }
    }
});

async function startSequence(urls) {
    await clearDB();
    await chrome.alarms.clearAll();
    const urlStatuses = urls.map(url => ({ url: url, status: 'pending', summary: '' }));
    await chrome.storage.local.set({
        isProcessing: true,
        urlsQueue: urls,
        totalUrls: urls.length,
        urlStatuses: urlStatuses,
        processingStatus: 'Starting...',
        pausedUrl: null
    });
    processNextUrl();
}

async function processNextUrl() {
    const state = await chrome.storage.local.get(['isProcessing', 'urlsQueue']);
    if (!state.isProcessing) return;

    const queue = state.urlsQueue || [];
    if (queue.length > 0) {
        const url = queue.shift();
        const { totalUrls } = await chrome.storage.local.get('totalUrls');
        const processedCount = totalUrls - queue.length;
        const statusMessage = `Processing ${processedCount} of ${totalUrls}...`;

        await chrome.storage.local.set({ urlsQueue: queue, processingStatus: statusMessage });
        await updateUrlStatus(url, 'processing', 'Scraping page...');

        let tabId = await getTabId();
        try {
            if (tabId) await chrome.tabs.get(tabId);
            else throw new Error('Tab not found');
            await chrome.tabs.update(tabId, { url: url, active: true });
        } catch (e) {
            const tab = await chrome.tabs.create({ url: url, active: true });
            await setTabId(tab.id);
        }
    } else {
        finalizeSequence(false);
    }
}

async function finalizeSequence(isManualStop) {
    const { isProcessing } = await chrome.storage.local.get('isProcessing');
    if (!isProcessing) return;

    await chrome.storage.local.set({ processingStatus: 'Finalizing... Generating CSV.' });
    const allData = await getAllDataFromDB();

    if (allData && allData.length > 0) {
        const csvData = convertToFinalCSV(allData);
        const blob = new Blob(["\uFEFF" + csvData], { type: 'text/csv;charset=utf-8;' });
        const reader = new FileReader();
        reader.onload = function() {
            chrome.downloads.download({
                url: reader.result,
                filename: 'shopee_product_review_bulk_scraper.csv',
                saveAs: false
            });
        };
        reader.readAsDataURL(blob);
    }

    await chrome.storage.local.set({ isProcessing: false, processingStatus: 'Finished! See summary for details.' });
    await chrome.alarms.clearAll();

    if (!isManualStop) {
        setTimeout(async () => {
            let tabId = await getTabId();
            if (tabId) { try { await chrome.tabs.remove(tabId); } catch (e) {} }
        }, 5000);
    }
}
