// This content script is the 'worker' on each product page.

// =================================================================================
// --- 1. Helper Functions ---
// =================================================================================

function waitForElement(selector, timeout = 15000) {
    return new Promise((resolve, reject) => {
        const intervalTime = 500;
        let elapsedTime = 0;
        const interval = setInterval(() => {
            const element = document.querySelector(selector);
            if (element) {
                clearInterval(interval);
                resolve(element);
            } else {
                elapsedTime += intervalTime;
                if (elapsedTime >= timeout) {
                    clearInterval(interval);
                    reject(new Error(`Timeout: Element "${selector}" not found.`));
                }
            }
        }, intervalTime);
    });
}

// =================================================================================
// --- 2. DATA INTERCEPTION ---
// =================================================================================

let pageProductData = { name: 'not_found', shop: 'not_found', itemid: null, shopid: null };

function injectScript(filePath) {
    const s = document.createElement('script');
    s.src = chrome.runtime.getURL(filePath);
    s.onload = function() { this.remove(); };
    (document.head || document.documentElement).appendChild(s);
}
injectScript('interceptor.js');

window.addEventListener('ProductDataIntercepted', (event) => {
    try {
        const data = JSON.parse(event.detail);
        if (data.data && data.data.item && data.data.shop_detailed) {
            pageProductData.name = data.data.item.title;
            pageProductData.shop = data.data.shop_detailed.name;
            pageProductData.itemid = data.data.item.item_id;
            pageProductData.shopid = data.data.item.shopid;
        }
    } catch (e) {
        console.error('[Bulk Scraper] Failed to process product data.', e);
    }
});

// =================================================================================
// --- 3. MAIN LOGIC ---
// =================================================================================

window.addEventListener('load', () => {
    chrome.storage.local.get('isProcessing', (result) => {
        if (result.isProcessing) navigateAndScrapeReviews();
    });
});

async function navigateAndScrapeReviews() {
    const pageNavigationDelay = 5000;
    const delay = ms => new Promise(resolve => setTimeout(resolve, ms));
    let singleProductReviews = [];
    const currentUrl = window.location.href;

    try {
        console.log(`[Bulk Scraper] Script loaded for ${currentUrl}.`);

        const reviewListener = (event) => {
            try {
                const data = JSON.parse(event.detail);
                if (data.data && data.data.ratings) {
                    console.log(`[Bulk Scraper] Intercepted ${data.data.ratings.length} reviews.`);
                    singleProductReviews.push(...data.data.ratings);
                }
            } catch (e) {
                console.error('[Bulk Scraper] Failed to process review data.', e);
            }
        };
        window.addEventListener('ReviewDataIntercepted', reviewListener);

        await waitForElement('.product-rating-overview__filter');
        console.log(`[Bulk Scraper] Discarding ${singleProductReviews.length} initial reviews.`);
        singleProductReviews = [];

        const filterButtons = document.querySelectorAll('.product-rating-overview__filter');
        let mediaFilterButton = Array.from(filterButtons).find(b => b.textContent.includes('Dengan Media') || b.textContent.includes('With Media'));
        if (mediaFilterButton && !mediaFilterButton.classList.contains('product-rating-overview__filter--active')) {
            console.log('[Bulk Scraper] Clicking "Dengan Media" button...');
            mediaFilterButton.click();
            await delay(pageNavigationDelay);
        }

        let lastKnownPage = -1;
        while (true) {
            const container = document.querySelector('.shopee-page-controller.product-ratings__page-controller');
            if (!container) break;
            const activeButton = container.querySelector('.shopee-button-solid--primary');
            const currentPage = activeButton ? parseInt(activeButton.innerText, 10) : 0;
            if (currentPage !== 0 && lastKnownPage === currentPage) break;
            lastKnownPage = currentPage;
            const nextPage = currentPage + 1;
            let nextPageButton = Array.from(container.querySelectorAll('.shopee-button-no-outline')).find(b => parseInt(b.innerText, 10) === nextPage);
            if (nextPageButton) {
                nextPageButton.click();
                await delay(pageNavigationDelay);
            } else {
                const nextArrowButton = container.querySelector('.shopee-icon-button--right');
                if (nextArrowButton && !nextArrowButton.hasAttribute('disabled')) {
                    nextArrowButton.click();
                    await delay(pageNavigationDelay);
                } else {
                    break;
                }
            }
        }

        window.removeEventListener('ReviewDataIntercepted', reviewListener);

        if (singleProductReviews.length > 0) {
            let imageCount = 0, videoCount = 0;
            singleProductReviews.forEach(r => {
                if (r.images && r.images.length > 0) imageCount++;
                if (r.videos && r.videos.length > 0) videoCount++;
            });
            const summary = `Success: ${singleProductReviews.length} reviews found (${imageCount} with images, ${videoCount} with videos).`;

            const newData = singleProductReviews.map(review => ({ productName: pageProductData.name, shopName: pageProductData.shop, productUrl: currentUrl, review: review }));
            chrome.runtime.sendMessage({ action: 'scraped_reviews_for_db', data: newData });
            chrome.runtime.sendMessage({ action: 'url_processing_finished', payload: { status: 'success', url: currentUrl, summary: summary } });
        } else {
            // No reviews found after scraping, create and send a placeholder.
            console.log('[Bulk Scraper] No reviews found after scraping. Creating placeholder entry.');
            const placeholderReview = {
                productName: pageProductData.name,
                shopName: pageProductData.shop,
                productUrl: currentUrl,
                review: {
                    cmtid: pageProductData.itemid, // Use itemid as a unique fake cmtid
                    itemid: pageProductData.itemid,
                    shopid: pageProductData.shopid,
                    author_username: 'system',
                    comment: '[PRODUK TIDAK MEMILIKI ULASAN]',
                    rating_star: 0,
                    product_items: [{ modelid: 0, model_name: 'N/A' }],
                    media_type: 'no_review'
                }
            };
            chrome.runtime.sendMessage({ action: 'scraped_reviews_for_db', data: [placeholderReview] });
            chrome.runtime.sendMessage({ action: 'url_processing_finished', payload: { status: 'success', url: currentUrl, summary: 'Success: 0 reviews found' } });
        }

    } catch (error) {
        console.error(`[Bulk Scraper] An error occurred on ${currentUrl}:`, error);
        if (error.message.includes('Timeout')) {
            // This is now more likely to be a real CAPTCHA or page error
            chrome.runtime.sendMessage({ 
                action: 'pause_for_captcha', 
                payload: { url: currentUrl, summary: 'Action Required: Possible CAPTCHA or page error. Please check the tab.' } // Updated summary
            });
        } else {
            // For other unexpected errors, fail the URL and continue the queue
            chrome.runtime.sendMessage({ 
                action: 'url_processing_finished', 
                payload: { status: 'failed', url: currentUrl, summary: `Error: ${error.message}` }
            });
        }
    }
}