document.addEventListener('DOMContentLoaded', () => {
    const startButton = document.getElementById('startButton');
    const finishButton = document.getElementById('finishButton');
    const exportJsonButton = document.getElementById('exportJsonButton');
    const openDashboardButton = document.getElementById('openDashboardButton');
    const urlsTextarea = document.getElementById('urls');
    const statusDiv = document.getElementById('status');
    const summaryLogDiv = document.getElementById('summary-log');
    const finalSummaryContainer = document.getElementById('final-summary-container');
    const finalSummaryTextarea = document.getElementById('final-summary');

    const renderSummary = (statuses) => {
        if (!statuses || statuses.length === 0) {
            summaryLogDiv.innerHTML = 'Waiting to start...';
            return;
        }

        let html = '<ul>';
        statuses.forEach(item => {
            let icon = '⏳';
            if (item.status === 'success') icon = '✅';
            if (item.status === 'failed') icon = '❌';
            if (item.status === 'processing') icon = '🏃';

            html += `<li><strong>${icon} [${item.status}]</strong>: ${item.summary || ''}<br><small>${item.url}</small></li>`;
        });
        html += '</ul>';
        summaryLogDiv.innerHTML = html;
    };

    const checkStatus = () => {
        chrome.storage.local.get(['isProcessing', 'processingStatus', 'urlStatuses'], (result) => {
            const isProcessing = result.isProcessing || false;
            const status = result.processingStatus || 'Ready.';

            // Update button states
            startButton.disabled = isProcessing;
            exportJsonButton.disabled = isProcessing;
            finishButton.disabled = !isProcessing;
            urlsTextarea.disabled = isProcessing;
            statusDiv.textContent = status;

            // Render the log
            renderSummary(result.urlStatuses);

            // Handle final summary visibility
            if (!isProcessing && result.urlStatuses && result.urlStatuses.length > 0) {
                const summaryText = result.urlStatuses.map(item => 
                    `[${item.status.toUpperCase()}] - ${item.summary}\nURL: ${item.url}`
                ).join('\n\n');
                finalSummaryTextarea.value = summaryText;
                finalSummaryContainer.style.display = 'block';
            } else {
                finalSummaryContainer.style.display = 'none';
            }
        });
    };

    startButton.addEventListener('click', () => {
        const urls = urlsTextarea.value.split('\n').map(url => url.trim()).filter(url => url.length > 0 && url.includes('shopee.'));
        if (urls.length === 0) {
            statusDiv.textContent = 'Please paste at least one valid Shopee product URL.';
            return;
        }
        chrome.runtime.sendMessage({ action: 'start_sequence', urls: urls });
    });

    finishButton.addEventListener('click', () => {
        chrome.runtime.sendMessage({ action: 'stop_sequence' });
    });

    exportJsonButton.addEventListener('click', () => {
        chrome.runtime.sendMessage({ action: 'export_db_as_json' });
    });

    openDashboardButton.addEventListener('click', () => {
        chrome.tabs.create({ url: 'popup.html' });
    });

    // Listen for live updates from the background script
    chrome.storage.onChanged.addListener((changes, namespace) => {
        if (namespace === 'local') {
            checkStatus(); // Re-render everything on any storage change
        }
    });

    // Initial status check when popup is opened
    checkStatus();
});