let isAutoScraping = false;
let autoScrapeInterval = null;
const AUTO_SCRAPE_INTERVAL = 4000; // 4 seconds

// --- Main Message Listener ---
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    switch (request.action) {
        case 'toggle_auto_scrape':
            toggleAutoScrape();
            break;
        case 'get_status':
            sendResponse({ isScraping: isAutoScraping });
            break;
        case 'status_update': // This case is handled by the popup, not the background
            break;
        case 'store_products':
            storeProducts(request.payload);
            break;
        case 'next_page':
            // Manual next page click
            chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
                if (tabs && tabs.length > 0) {
                    goToNextPage(tabs[0].id);
                }
            });
            break;
        case 'finish':
            generateAndDownloadCSV();
            break;
        case 'clear':
            clearStorage();
            break;
    }
    return true; // Indicates asynchronous response
});

// --- Auto-Scraping Logic ---

function toggleAutoScrape() {
    if (isAutoScraping) {
        stopAutoScrape();
    } else {
        startAutoScrape();
    }
}

function startAutoScrape() {
    isAutoScraping = true;
    notifyUIStatusChange();
    console.log('Auto-scraping started.');

    // Perform the first click immediately
    performAutoClick();

    // Set up the interval for subsequent clicks
    autoScrapeInterval = setInterval(performAutoClick, AUTO_SCRAPE_INTERVAL);
}

function stopAutoScrape() {
    isAutoScraping = false;
    if (autoScrapeInterval) {
        clearInterval(autoScrapeInterval);
        autoScrapeInterval = null;
    }
    notifyUIStatusChange();
    console.log('Auto-scraping stopped.');
}

function performAutoClick() {
    if (!isAutoScraping) {
        stopAutoScrape();
        return;
    }

    console.log('Auto-scrape: Triggering next page action...');
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        if (tabs && tabs.length > 0) {
            goToNextPage(tabs[0].id, (success) => {
                if (!success) {
                    console.log('Auto-scraping finished: No more pages found. Starting download.');
                    stopAutoScrape();
                    generateAndDownloadCSV();
                }
                // If successful, the loop will continue with the next interval.
            });
        } else {
            console.error('Auto-scrape: No active tab found. Stopping.');
            stopAutoScrape();
        }
    });
}

function notifyUIStatusChange() {
    chrome.runtime.sendMessage({ action: 'status_update', isScraping: isAutoScraping });
}

// --- Core Functions (Refactored) ---

async function storeProducts(newProducts) {
    const { scrapedProducts = [] } = await chrome.storage.local.get('scrapedProducts');
    
    if (scrapedProducts.length > 0 && newProducts.length > 0) {
        const lastStoredLink = scrapedProducts[scrapedProducts.length - 1].product_link;
        const firstNewLink = newProducts[0].product_link;
        if (lastStoredLink === firstNewLink) {
            console.log("Shop Bulk Scraper: Data from this page seems to be already stored. Skipping.");
            return;
        }
    }

    const updatedProducts = [...scrapedProducts, ...newProducts];
    await chrome.storage.local.set({ scrapedProducts: updatedProducts });
    console.log(`Shop Bulk Scraper: Stored ${newProducts.length} new products. Total: ${updatedProducts.length}`);
}

function goToNextPage(tabId, callback) {
    chrome.scripting.executeScript(
        {
            target: { tabId: tabId },
            function: () => {
                let isOnLastPage = false;
                const pageController = document.querySelector('.shopee-page-controller');
                if (pageController) {
                    const activeButton = pageController.querySelector('.shopee-button-solid.shopee-button-solid--primary');
                    if (activeButton) {
                        const currentPage = parseInt(activeButton.innerText, 10);
                        if (!isNaN(currentPage)) {
                            const nextPage = currentPage + 1;
                            let nextPageFound = false;
                            const allButtons = pageController.querySelectorAll('button');
                            for (const btn of allButtons) {
                                if (parseInt(btn.innerText, 10) === nextPage) {
                                    nextPageFound = true;
                                    break;
                                }
                            }
                            if (!nextPageFound) {
                                isOnLastPage = true;
                            }
                        }
                    } else {
                        const nextArrow = pageController.querySelector('.shopee-icon-button--right');
                        if (nextArrow && nextArrow.classList.contains('shopee-icon-button--disabled')) {
                            isOnLastPage = true;
                        }
                    }
                }

                const nextArrowButton = document.querySelector('.shopee-icon-button--right');
                if (nextArrowButton && !nextArrowButton.classList.contains('shopee-icon-button--disabled') && !isOnLastPage) {
                    nextArrowButton.click();
                    return true; // Action taken
                }

                const outlineButtons = Array.from(document.querySelectorAll('button.shopee-button-outline'));
                const soldOutButton = outlineButtons.find(btn => btn.innerText.includes('See More') || btn.innerText.includes('Lihat Lainnya'));
                if (soldOutButton) {
                    soldOutButton.click();
                    return true; // Action taken
                }

                // Fallback for the original selector
                const oldSoldOutButton = document.querySelector('.shop-sold-out-see-more button');
                if (oldSoldOutButton) {
                    oldSoldOutButton.click();
                    return true; // Action taken
                }
                
                const isAtBottom = window.innerHeight + window.scrollY >= document.body.scrollHeight - 10;
                if (!isAtBottom) {
                    window.scrollTo(0, document.body.scrollHeight);
                    return true; // Action taken
                }

                return false; // No action taken, end of pages
            },
        },
        (injectionResults) => {
            if (callback) {
                if (chrome.runtime.lastError) {
                    console.error(chrome.runtime.lastError);
                    callback(false);
                    return;
                }
                if (injectionResults && injectionResults[0]) {
                    callback(injectionResults[0].result);
                } else {
                    callback(false);
                }
            }
        }
    );
}

async function generateAndDownloadCSV() {
    const { scrapedProducts = [] } = await chrome.storage.local.get('scrapedProducts');
    if (scrapedProducts.length === 0) {
        console.warn("Shop Bulk Scraper: No products to download.");
        return;
    }

    const headers = ['ItemID', 'ShopID', 'ModelID', 'Name', 'Shop Name', 'Price', 'Sold', 'Rating', 'Rating Count', 'Liked Count', 'Shop Location', 'Brand', 'Variations', 'Product Link', 'Image URL'];
    const csvRows = [headers.join(';')];

    scrapedProducts.forEach(p => {
        const values = [
            p.itemid,
            p.shopid,
            p.model_id,
            `"${p.name.replaceAll('"', '""')}"`, // Corrected escaping for double quotes within strings
            `"${p.shop_name.replaceAll('"', '""')}"`, // Corrected escaping for double quotes within strings
            p.price,
            p.sold,
            p.rating,
            p.rating_count,
            p.liked_count,
            `"${(p.shop_location || '').replaceAll('"', '""')}"`, // Corrected escaping for double quotes within strings
            `"${(p.brand || '').replaceAll('"', '""')}"`, // Corrected escaping for double quotes within strings
            `"${(p.variations || '').replaceAll('"', '""')}"`, // Corrected escaping for double quotes within strings
            `"${p.product_link}"`, // Corrected escaping for double quotes within strings
            `"${p.image_url}"` // Corrected escaping for double quotes within strings
        ];
        csvRows.push(values.join(';'));
    });

    const csvContent = csvRows.join('\n');
    const blob = new Blob(["\uFEFF" + csvContent], { type: 'text/csv;charset=utf-8;' });
    
    const reader = new FileReader();
    reader.onload = function() {
        const dataUrl = reader.result;
        const shopName = scrapedProducts[0].shop_name.replace(/[/\?%*:|"<>]/g, '-');
        const now = new Date();
        const timestamp = `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(2, '0')}-${String(now.getDate()).padStart(2, '0')}_${String(now.getHours()).padStart(2, '0')}-${String(now.getMinutes()).padStart(2, '0')}-${String(now.getSeconds()).padStart(2, '0')}`;
        const filename = `etalase toko - ${shopName} - ${timestamp}.csv`;

        chrome.downloads.download({
            url: dataUrl,
            filename: filename,
            saveAs: true
        });
    };
    reader.readAsDataURL(blob);

    clearStorage();
}

function clearStorage() {
    chrome.storage.local.set({ scrapedProducts: [] });
    console.log("Shop Bulk Scraper: Storage cleared.");
}