const countElement = document.getElementById('productCount');
const autoScrapeBtn = document.getElementById('autoScrapeBtn');
const nextPageBtn = document.getElementById('nextPageBtn');
const finishBtn = document.getElementById('finishBtn');
const clearBtn = document.getElementById('clearBtn');

// --- Event Listeners ---
autoScrapeBtn.addEventListener('click', () => {
  chrome.runtime.sendMessage({ action: 'toggle_auto_scrape' });
});

nextPageBtn.addEventListener('click', () => {
  chrome.runtime.sendMessage({ action: 'next_page' });
});

finishBtn.addEventListener('click', () => {
  chrome.runtime.sendMessage({ action: 'finish' });
});

clearBtn.addEventListener('click', () => {
  chrome.runtime.sendMessage({ action: 'clear' });
});

// --- UI Update Functions ---

// This function now also handles the initial enabling of buttons
function updateCountAndButtonState() {
  chrome.storage.local.get('scrapedProducts', (result) => {
    const products = result.scrapedProducts || [];
    countElement.textContent = products.length;

    // Only enable action buttons after the first scrape is successful
    if (products.length > 0) {
        // Check if auto-scraping is active before enabling manual buttons
        chrome.runtime.sendMessage({ action: 'get_status' }, (response) => {
            if (!response || !response.isScraping) {
                enableActionButtons();
            }
        });
    } else {
        disableActionButtons();
    }
  });
}

function disableActionButtons() {
    autoScrapeBtn.disabled = true;
    nextPageBtn.disabled = true;
    autoScrapeBtn.textContent = 'Waiting for products...';
}

function enableActionButtons() {
    autoScrapeBtn.disabled = false;
    nextPageBtn.disabled = false;
    autoScrapeBtn.textContent = 'Auto Scrape All 🚀';
}

function setAutoScrapeUI(isScraping) {
    if (isScraping) {
        autoScrapeBtn.textContent = 'Stop Auto Scrape 🛑';
        autoScrapeBtn.style.backgroundColor = '#c62828'; // Red color for stop
        nextPageBtn.disabled = true; // Disable manual next while auto is running
        finishBtn.disabled = true;
        clearBtn.disabled = true;
    } else {
        autoScrapeBtn.textContent = 'Auto Scrape All 🚀';
        autoScrapeBtn.style.backgroundColor = '#ff6f00'; // Orange color for start
        // Re-enable buttons only if products have been scraped
        updateCountAndButtonState();
        finishBtn.disabled = false;
        clearBtn.disabled = false;
    }
}

// --- Initial Setup and Listeners ---

// Ask background for current auto-scrape status when popup opens
chrome.runtime.sendMessage({ action: 'get_status' }, (response) => {
    if (chrome.runtime.lastError) {
        console.log("Could not get status from background.");
        setAutoScrapeUI(false);
    } else {
        setAutoScrapeUI(response.isScraping);
    }
});

// Listen for storage changes to update the product count and button states
chrome.storage.onChanged.addListener((changes, namespace) => {
  if (namespace === 'local' && changes.scrapedProducts) {
    updateCountAndButtonState();
  }
});

// Listen for status update messages from the background script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'status_update') {
        setAutoScrapeUI(request.isScraping);
    }
});

// Initial count and button state update when popup is opened
updateCountAndButtonState();